//**************************************************************************************************
//                                        PrcDataVwr.cpp                                           *
//                                       ----------------                                          *
// Started     : 2020-09-26                                                                        *
// Last Update : 2020-10-22                                                                        *
// Copyright   : (C) 2020-2020 MSWaters                                                            *
//**************************************************************************************************

//**************************************************************************************************
//                                                                                                 *
//      This program is free software; you can redistribute it and/or modify it under the          *
//      terms of the GNU General Public License as published by the Free Software Foundation;      *
//      either version 3 of the License, or (at your option) any later version.                    *
//                                                                                                 *
//**************************************************************************************************

#include "PrcDataVwr.hpp"

//**************************************************************************************************
// Constructor.

PrcDataVwr::PrcDataVwr( void ) : PrcBase( wxPROCESS_REDIRECT )
{
  // Initialize the object attributes
//  m_eDataVwr = eDVWR_NONE;
  m_ofnNameResults.Clear( );

  // Attempt to find and set the binary file
  bSetDataViewer( g_oConfig.eGetDataViewer( ) );
}

//**************************************************************************************************
// Destructor.

PrcDataVwr::~PrcDataVwr( )
{
}

//**************************************************************************************************
// Set the data viewer application type.
//
// Argument List:
//  eDataVwr - an enumerated type indicating the data viewer application
//
// Return Values:
//   true  - Success
//   false - Failure

bool  PrcDataVwr::bSetDataViewer( eTypeDatVwr eDataVwr )
{
  bool  bRtn;

  if( m_eDataVwr==eDataVwr && PrcBase::bBinExists( ) ) return( true );

  switch( eDataVwr )
  {
    case eDVWR_GAW   : bRtn = bSetBinFile( BIN_GAW   ); break;
    case eDVWR_GWAVE : bRtn = bSetBinFile( BIN_GWAVE ); break;
    case eDVWR_KST   : bRtn = bSetBinFile( BIN_KST   ); break;
    default          : return( false );
  }

  if( bRtn ) m_eDataVwr = eDataVwr;

  return( bRtn );
}

//**************************************************************************************************
// Set the simulation results file name.
//
// Argument List:
//   psFileName - a string containing the full path and file name
//
// Return Values:
//   true  - Success
//   false - Failure

bool  PrcDataVwr::bSetResultsFile( const wxString & rosFileName )
{
  wxFileName  ofn1;

  ofn1 = rosFileName;
  if( ofn1.GetPath( ).IsEmpty( ) ) ofn1.SetPath( "." );

  if( ! ofn1.IsOk( )       ) return( false );
  if( ! ofn1.FileExists( ) ) return( false );

  m_ofnNameResults = ofn1;

  return( true );
}

//**************************************************************************************************
// Format the results file so that the data viewer utility is happy with it.
//
// There are several things that a data viewer can object to :
//  - gwave doesn't like the GNU-CAP banner at the start to the results file, so it's removed here.
//  - gwave doesn't like error messages interspersed with the lines of data, so they are removed
//    here.
//  - For good measure lines and fields are also formatted.
//
// Return Values:
//   true  - Success
//   false - Failure

bool  PrcDataVwr::bFormatFile( void )
{
  wxTextFile  oFileResults;
  wxString    os1;
  size_t      sz1, sz2;
  wxChar      oc1;
  bool        bRtn=true;

  // Attempt to open the results file
  if( ! m_ofnNameResults.FileExists( ) )                       return( false );
  if( ! oFileResults.Open( m_ofnNameResults.GetFullPath( ) ) ) return( false );

  // Has this file been formatted already?
  if( oFileResults.GetFirstLine( ).GetChar( 0 ) == '#' )       return( true );

  // Find the beginning of the data area (ie. the last line beginning with '#')
  for( sz1=sz2=0; sz1<oFileResults.GetLineCount( ); sz1++ )
  {
    os1 = oFileResults.GetLine( sz1 );
    if( ! os1.IsEmpty( ) )
      if( os1.GetChar( 0 ) == '#' )
        sz2 = sz1;
  }

  // Delete the banner
  for( sz1=0; sz1<sz2; sz1++ ) oFileResults.RemoveLine( 0 );
  if( oFileResults.GetLineCount( ) <= 1 )                      return( false );

  // Delete any simulator error messages eg. "open circuit: internal node 3"
  // (All lines should start with a digit except for the first)
  for( sz1=1; !oFileResults.Eof( ) && sz1<oFileResults.GetLineCount( ); sz1++ )
  {
    os1 = oFileResults.GetLine( sz1 );
    if( os1.Length( ) <= 1 )
    { // Delete empty lines
      oFileResults.RemoveLine( sz1 );
      sz1--;
      continue;
    }
    oc1 = os1.GetChar( 1 );
    if( ! wxIsdigit( oc1 ) )
    { // Delete non-data lines
      oFileResults.RemoveLine( sz1 );
      sz1--;
      continue;
    }
  }

  // Format each data line in the file
//  for( sz1=1; sz1<oFileResults.GetLineCount( ); sz1++ )
//    if( ! bFormatLine( oFileResults.GetLine( sz1 ) ) )
//      bRtn = false;

  oFileResults.Write( ); // Save the changes to disk
  oFileResults.Close( ); // Close the file

  return( bRtn );
}

//**************************************************************************************************
// Reformat the lines from the results file.
//
// Argument List:
//   rosLine - The line to be formatted
//
// Return Values:
//   true  - Success
//   false - Failure
/*
bool  PrcDataVwr::bFormatLine( wxString & rosLine )
{
  wxStringTokenizer  ostk1;
  wxString  os1, os2;

  // Check for an empty string
  if( rosLine.IsEmpty( ) )       return( false );

  // Break the line into fields
  ostk1.SetString( rosLine );
  if( ostk1.CountTokens( ) < 2 ) return( false );

  // Reformat the line
  while( ostk1.HasMoreTokens( ) )
  {
    os1 = ostk1.GetNextToken( );
    if( ! bFormatField( os1 ) )  return( false );
    if( os2.IsEmpty( ) ) os2 = os1;
    else                 os2 << "     " << os1;
  }

  rosLine = os2;

  return( true );
}
*/
//**************************************************************************************************
// Reformat the fields from the results file.
//
// Argument List:
//   rosField - The field to be formatted
//
// Return Values:
//   true  - Success
//   false - Failure
/*
bool  PrcDataVwr::bFormatField( wxString & rosField )
{
  wxString  os1;
  wxChar    oc1;
  size_t    szt1;

  // Check for an empty string
  if( rosField.IsEmpty( ) ) return( false );

  // Extract the value and the units
  oc1 = 0;
  for( szt1=0; szt1<rosField.Length( ); szt1++ )
  {
    oc1 = rosField.GetChar( szt1 );
    if( oc1!='-' && !wxIsdigit( oc1 ) && oc1!='.' ) break;
    else oc1 = 0;
  }
  os1 = rosField.Left( szt1 );

  // Reformat the field
  switch( oc1 )
  {
    case 'M' : os1 << "E+06"; break;
    case 'K' : os1 << "E+03"; break;
    case 'm' : os1 << "E-03"; break;
    case 'u' : os1 << "E-06"; break;
    case 'n' : os1 << "E-09"; break;
    case 'p' : os1 << "E-12"; break;
    case 0   : break;
    default  : return( false );
  }

  rosField = os1;

  return( true );
}
*/
//**************************************************************************************************
// View the results of a simulation.
//
// (Eg. using the following: Kst ../sch/test-amp1.gnucap.dc)
//
// Return Values:
//   true  - Success
//   false - Failure

bool  PrcDataVwr::bExec( void )
{
  wxTextFile         oFileResults;
  wxStringTokenizer  ostk1;
  wxString           os1;

  // Check that the results data file has been set
  if( !m_ofnNameResults.FileExists( ) || !m_ofnNameResults.IsOk( ) )
  {
    os1 = m_ofnNameResults.GetFullPath( );
    PrcBase::ClrErrMsg( );
    if( os1.IsEmpty( ) ) PrcBase::SetErrMsg( "The results file name has not been set." );
    else                 PrcBase::SetErrMsg( "The results file doesn't exist : " + os1 );
    return( false );
  }

  // Add the results file name to argument list
  os1 = m_ofnNameResults.GetFullPath( );

  // Create data viewer application specific arguments (if any)
  switch( m_eDataVwr )
  {
    case eDVWR_GAW   : break;
    case eDVWR_GWAVE : break;

    case eDVWR_KST   :
      // Attempt to open the results file and create the Kst specific argument list
      if( oFileResults.Open( m_ofnNameResults.GetFullPath( ) ) )
      {
        // The first line contains column labels, use them to create graph names
        ostk1.SetString( oFileResults.GetFirstLine( ) );

        // Start with the stardard arguments
        os1 << " --asciiDataStart 2 --asciiFieldNames 1 --asciiNoUnits --asciiSpaceDelim"
            << " --asciiDecimalDot";

        // Add a graph for each parameter in the results file, using the first column as X axes
        os1 << " -x 1 --xlabel '" << ostk1.GetNextToken( ).substr( 1 ) << "'";
        for( int i1=2; ostk1.HasMoreTokens( ); i1++ )
          os1 << " --ylabel '" << ostk1.GetNextToken( ) << "' -y " << i1;

        // Close the data file
        oFileResults.Close( );
      }
      break;

    default          : return( false );
  }

  // Set the process argument list
  os1 = rosEscSpaceChrs( os1 );
  if( ! bSetArgLst( os1 ) )      return( false );

  // Format the file before passing it to the waveform viewer process
  if( ! bFormatFile( ) )         return( false );

  // Execute the process
  if( ! PrcBase::bExecAsync( ) ) return( false );

  return( true );
}

//**************************************************************************************************
